<?php
/*
Plugin Name: Pdee Events
Description: Manage and display custom events with shortcode.
Version: 1.0.1
Author: Pdee
*/

// 1. Register Custom Post Type
function pdee_register_events_post_type() {
    register_post_type('pdee_events',
        array(
            'labels' => array(
                'name' => __('Pdee Events'),
                'singular_name' => __('Pdee Event')
            ),
            'public' => true,
            'has_archive' => true,
            'menu_icon' => 'dashicons-calendar-alt',
            'supports' => array('title', 'editor')
        )
    );
}
add_action('init', 'pdee_register_events_post_type');

// 2. Add Meta Boxes for Extra Fields
function pdee_add_event_metaboxes() {
    add_meta_box('pdee_event_details', 'Event Details', 'pdee_render_metabox', 'pdee_events', 'normal', 'default');
}
add_action('add_meta_boxes', 'pdee_add_event_metaboxes');

function pdee_render_metabox($post) {
    // Get current values
    $day = get_post_meta($post->ID, 'pdee_day', true);
    $month = get_post_meta($post->ID, 'pdee_month', true);
    $year = get_post_meta($post->ID, 'pdee_year', true);
    $url = get_post_meta($post->ID, 'pdee_event_url', true);

    // Form
    ?>
    <label>Day:</label><br>
    <input type="text" name="pdee_day" value="<?php echo esc_attr($day); ?>" style="width: 100%;"><br><br>

    <label>Month:</label><br>
    <input type="text" name="pdee_month" value="<?php echo esc_attr($month); ?>" style="width: 100%;"><br><br>

    <label>Year:</label><br>
    <input type="text" name="pdee_year" value="<?php echo esc_attr($year); ?>" style="width: 100%;"><br><br>

    <label>Event URL:</label><br>
    <input type="url" name="pdee_event_url" value="<?php echo esc_url($url); ?>" style="width: 100%;"><br>
    <?php
}

// 3. Save Meta Fields
function pdee_save_event_meta($post_id) {
    if (defined('DOING_AUTOSAVE') && DOING_AUTOSAVE) return;

    if (isset($_POST['pdee_day']))
        update_post_meta($post_id, 'pdee_day', sanitize_text_field($_POST['pdee_day']));

    if (isset($_POST['pdee_month']))
        update_post_meta($post_id, 'pdee_month', sanitize_text_field($_POST['pdee_month']));

    if (isset($_POST['pdee_year']))
        update_post_meta($post_id, 'pdee_year', sanitize_text_field($_POST['pdee_year']));

    if (isset($_POST['pdee_event_url']))
        update_post_meta($post_id, 'pdee_event_url', esc_url_raw($_POST['pdee_event_url']));
}
add_action('save_post', 'pdee_save_event_meta');

// 4. Shortcode to Display Events  – UPCOMING FIRST
function pdee_events_shortcode( $atts = array() ) {
    ob_start();

    // ----------  STYLES  (print once) ----------
    static $pdee_styles_printed = false;
    if ( ! $pdee_styles_printed ) {
        ?>
        <style>
            .pdee-events       { display:flex; flex-direction:column; gap:20px; padding:10px; }
            .pdee-event        { border:2px solid #ccc; border-radius:10px; padding:15px;
                                 display:flex; gap:15px; background:#f9f9f9; align-items:flex-start; }
            .pdee-event-date   { background:#003366; color:#fff; font-size:24px; font-weight:800;
                                 padding:10px; min-width:100px; text-align:center; border-radius:8px; }
            .pdee-event-title  { font-size:20px; font-weight:bold; margin-bottom:8px; }
            .pdee-event-description { font-size:16px; color:#333; margin-bottom:10px; }
            .pdee-event-button { display:inline-block; background:#0073aa; color:#fff; padding:8px 16px;
                                 border-radius:5px; text-decoration:none; font-weight:bold; }
            .pdee-event-button:hover { background:#005077; }
        </style>
        <?php
        $pdee_styles_printed = true;
    }

    // ----------  GET ALL EVENTS ----------
    $events_query = new WP_Query( array(
        'post_type'      => 'pdee_events',
        'posts_per_page' => -1,            // get every event – we will sort in PHP
        'orderby'        => 'date',
        'order'          => 'ASC',
    ) );

    // Build two buckets: upcoming & past
    $upcoming = array();
    $past     = array();

    // Midnight today (site-timezone) to compare against
    $today_midnight = strtotime( date_i18n( 'Y-m-d' ) );

    while ( $events_query->have_posts() ) {
        $events_query->the_post();

        $day   = get_post_meta( get_the_ID(), 'pdee_day',   true );
        $month = get_post_meta( get_the_ID(), 'pdee_month', true );
        $year  = get_post_meta( get_the_ID(), 'pdee_year',  true );
        $url   = get_post_meta( get_the_ID(), 'pdee_event_url', true );

        // Turn Day / Month / Year into a Unix timestamp for easy comparison
        $event_ts = strtotime( "$day $month $year" );

        // Build a small array we can sort later
        $event_item = array(
            'ts'          => $event_ts,
            'day'         => $day,
            'month'       => $month,
            'year'        => $year,
            'title'       => get_the_title(),
            'description' => get_the_content(),
            'url'         => $url,
        );

        // Bucket it
        if ( $event_ts >= $today_midnight ) {
            $upcoming[] = $event_item;   // future (or today) → upcoming
        } else {
            $past[]     = $event_item;   // yesterday or earlier → past
        }
    }
    wp_reset_postdata();

    // ----------  SORT BUCKETS ----------
    // upcoming: earliest first
    usort( $upcoming, function ( $a, $b ) {
        return $a['ts'] <=> $b['ts'];
    } );
    // past: most-recent first
    usort( $past, function ( $a, $b ) {
        return $b['ts'] <=> $a['ts'];
    } );

    // ----------  OUTPUT ----------
    echo '<div class="pdee-events">';

    // Upcoming events first
    foreach ( array_merge( $upcoming, $past ) as $ev ) {
        ?>
        <div class="pdee-event">
            <div class="pdee-event-date">
                <?php echo esc_html( $ev['day'] ); ?><br>
                <?php echo esc_html( $ev['month'] ) . ' ' . esc_html( $ev['year'] ); ?>
            </div>

            <div class="pdee-event-details">
                <div class="pdee-event-title"><?php echo esc_html( $ev['title'] ); ?></div>
                <div class="pdee-event-description"><?php echo esc_html( $ev['description'] ); ?></div>
                <?php if ( $ev['url'] ) : ?>
                    <a class="pdee-event-button" href="<?php echo esc_url( $ev['url'] ); ?>" target="_blank">View Event</a>
                <?php endif; ?>
            </div>
        </div>
        <?php
    }

    echo '</div>';

    return ob_get_clean();
}
add_shortcode( 'pdee_events', 'pdee_events_shortcode' );
